const express = require("express");
const router = express.Router();
const ClientUsers = require("../models/clientUsers");
const auth = require("../middleware/auth");

// 🏗️ Create a new client user
router.post("/client/user/create", auth, async (req, res) => {
  try {
    const {
      client_id,
      loc_id,
      first_name,
      last_name,
      user_name,
      password,
      user_type,
      work_email,
      personal_email,
      telephone,
      user_status,
    } = req.body;

    if (!client_id || !loc_id || !user_name || !password) {
      return res.status(400).json({
        message: "client_id, loc_id, user_name, and password are required",
      });
    }

    const newUser = await ClientUsers.create({
      client_id,
      loc_id,
      first_name,
      last_name,
      user_name,
      password,
      user_type,
      work_email,
      personal_email,
      telephone,
      user_status,
    });

    res.status(201).json({
      status: true,
      message: "Client user created successfully",
      user: newUser,
    });
  } catch (err) {
    res
      .status(500)
      .json({ message: "Error creating client user", error: err.message });
  }
});

// 📋 Get all client users
router.get("/client/user/all", auth, async (req, res) => {
  try {
    const users = await ClientUsers.getAll();
    res.status(200).json({ status: true, users });
  } catch (err) {
    res
      .status(500)
      .json({ message: "Error fetching client users", error: err.message });
  }
});

// 🔍 Get users by client ID
router.get("/client/user/:client_id", auth, async (req, res) => {
  try {
    const { client_id } = req.params;
    const users = await ClientUsers.getByClientId(client_id);
    res.status(200).json({ status: true, users });
  } catch (err) {
    res
      .status(500)
      .json({ message: "Error fetching client users", error: err.message });
  }
});

// 🔄 Update client user
router.put("/client/user/update", auth, async (req, res) => {
  try {
    const {
      user_entity_id,
      client_id,
      loc_id,
      first_name,
      last_name,
      user_name,
      password,
      user_type,
      work_email,
      personal_email,
      telephone,
      user_status,
    } = req.body;

    if (!user_entity_id || !client_id || !loc_id) {
      return res.status(400).json({
        message: "user_entity_id, client_id, and loc_id are required",
      });
    }

    const updatedUser = await ClientUsers.update({
      user_entity_id,
      client_id,
      loc_id,
      first_name,
      last_name,
      user_name,
      password,
      user_type,
      work_email,
      personal_email,
      telephone,
      user_status,
    });

    res.status(200).json({
      status: true,
      message: "Client user updated successfully",
      user: updatedUser,
    });
  } catch (err) {
    res
      .status(500)
      .json({ message: "Error updating client user", error: err.message });
  }
});

// ❌ Delete client user
router.delete("/client/user/delete", auth, async (req, res) => {
  try {
    const { client_id, loc_id, user_entity_id } = req.body;

    // Ensure all necessary parameters are provided
    if (!client_id || !loc_id || !user_entity_id) {
      return res.status(400).json({
        message: "client_id, loc_id, and user_entity_id are required",
      });
    }

    // Perform deletion based on client_id, loc_id, and user_entity_id
    const result = await ClientUsers.delete(client_id, loc_id, user_entity_id);

    res.status(200).json({ status: true, message: result.message });
  } catch (err) {
    res.status(500).json({
      message: "Error deleting client user",
      error: err.message,
    });
  }
});

module.exports = router;
