const db = require("../../config/db");
const redis = require("../../config/redis");

class MasterShippingMethod {
  // 🟢 Create master_sales_channel Table
  static async createTable() {
    const query = `
      CREATE TABLE IF NOT EXISTS master_shipping_method (
        shipping_id SERIAL PRIMARY KEY,
        shipping_name VARCHAR(255) NOT NULL UNIQUE,
        url TEXT NOT NULL ,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      );
    `;

    try {
      await db.query(query);
      console.log("✅ master_shipping_method table is ready.");
    } catch (err) {
      console.error("❌ Error creating master_shipping_method table:", err);
    }
  }

  // 🟢 Create a new sales channel
  static async create({ shipping_name, url }) {
    const query = `
      INSERT INTO master_shipping_method (shipping_name, url)
      VALUES ($1, $2)
      RETURNING shipping_id;
    `;

    const values = [shipping_name, url];

    try {
      const { rows } = await db.query(query, values);
      await redis.del("shipping_methods:all"); // Invalidate cached list
      return rows[0].shipping_id;
    } catch (err) {
      console.error("❌ Error inserting shipping_methods:", err);
      throw err;
    }
  }

  // 🟢 Get all sales channels (with caching)
  static async getAll() {
    const cachedShipping = await redis.get("shipping_methods:all");
    if (cachedShipping) {
      return JSON.parse(cachedShipping);
    }

    const { rows } = await db.query("SELECT * FROM master_shipping_method");
    await redis.setex("shipping_methods:all", 3600, JSON.stringify(rows)); // Cache for 1 hour
    return rows;
  }

  // 🟢 Get a sales channel by ID (with caching)
  static async getById(shipping_id) {
    const cachedShipping = await redis.get(`shipping_methods:${shipping_id}`);
    if (cachedShipping) {
      return JSON.parse(cachedShipping);
    }

    const {
      rows,
    } = await db.query(
      "SELECT * FROM master_shipping_method WHERE shipping_id = $1",
      [shipping_id]
    );

    if (rows.length === 0) return null;

    await redis.setex(
      `shipping_methods:${shipping_id}`,
      3600,
      JSON.stringify(rows[0])
    );
    return rows[0];
  }

  // 🟢 Update a sales channel
  static async update({ shipping_id, shipping_name, url }) {
    const query = `
      UPDATE master_shipping_method
      SET shipping_name = $2, url = $3
      WHERE shipping_id = $1
      RETURNING shipping_id;
    `;

    const values = [shipping_id, shipping_name, url];

    try {
      const { rows } = await db.query(query, values);
      await redis.del(`shipping_methods:${shipping_id}`);
      await redis.del("shipping_methods:all");
      return rows[0].shipping_id;
    } catch (err) {
      console.error("❌ Error updating Shipping methods:", err);
      throw err;
    }
  }

  // 🟢 Delete a sales channel
  static async delete(shipping_id) {
    await db.query(
      "DELETE FROM master_shipping_method WHERE shipping_id = $1",
      [shipping_id]
    );
    await redis.del(`shipping_methods:${shipping_id}`);
    await redis.del("shipping_methods:all");
    return { message: "shipping_methods deleted successfully" };
  }
}

module.exports = MasterShippingMethod;
