const db = require("../../config/db");
const redis = require("../../config/redis");

class ClientTumblr {
  static async createTable() {
    const query = `
      CREATE TABLE IF NOT EXISTS client_tumblr (
        client_id INT PRIMARY KEY,
        sales_channel_id INT NOT NULL REFERENCES master_sales_channel(channel_id) ON DELETE CASCADE,
        consumer_key TEXT NOT NULL,
        consumer_secret TEXT NOT NULL,
        oauth_token TEXT NOT NULL,
        oauth_secret TEXT NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      );
    `;

    try {
      await db.query(query);
      console.log("✅ client_tumblr table is ready.");
    } catch (err) {
      console.error("❌ Error creating client_tumblr table:", err);
    }
  }

  static async create(clientData) {
    const query = `
      INSERT INTO client_tumblr (
        client_id, sales_channel_id, consumer_key, consumer_secret, oauth_token, oauth_secret
      ) VALUES ($1, $2, $3, $4, $5, $6)
      RETURNING client_id;
    `;

    const values = [
      clientData.client_id,
      clientData.sales_channel_id,
      clientData.consumer_key,
      clientData.consumer_secret,
      clientData.oauth_token,
      clientData.oauth_secret,
    ];

    try {
      const { rows } = await db.query(query, values);
      await redis.del("client_tumblr:all");
      return rows[0].client_id;
    } catch (err) {
      console.error("❌ Error inserting client record:", err);
      throw err;
    }
  }

  static async getAll() {
    const cachedClients = await redis.get("client_tumblr:all");
    if (cachedClients) {
      return JSON.parse(cachedClients);
    }

    try {
      const { rows } = await db.query("SELECT * FROM client_tumblr");
      await redis.setex("client_tumblr:all", 3600, JSON.stringify(rows));
      return rows;
    } catch (err) {
      console.error("❌ Error fetching clients:", err);
      throw err;
    }
  }

  static async getById(client_id) {
    const cachedClient = await redis.get(`client_tumblr:${client_id}`);
    if (cachedClient) {
      return JSON.parse(cachedClient);
    }

    try {
      const {
        rows,
      } = await db.query("SELECT * FROM client_tumblr WHERE client_id = $1", [
        client_id,
      ]);

      if (rows.length === 0) return null;

      await redis.setex(
        `client_tumblr:${client_id}`,
        3600,
        JSON.stringify(rows[0])
      );
      return rows[0];
    } catch (err) {
      console.error("❌ Error fetching client by ID:", err);
      throw err;
    }
  }

  static async update(clientData) {
    const query = `
      UPDATE client_tumblr
      SET sales_channel_id = $2, consumer_key = $3, consumer_secret = $4, 
          oauth_token = $5, oauth_secret = $6
      WHERE client_id = $1
      RETURNING client_id;
    `;

    const values = [
      clientData.client_id,
      clientData.sales_channel_id,
      clientData.consumer_key,
      clientData.consumer_secret,
      clientData.oauth_token,
      clientData.oauth_secret,
    ];

    try {
      const { rows } = await db.query(query, values);
      await redis.del(`client_tumblr:${clientData.client_id}`);
      await redis.del("client_tumblr:all");
      return rows[0].client_id;
    } catch (err) {
      console.error("❌ Error updating client record:", err);
      throw err;
    }
  }

  static async delete(client_id) {
    try {
      await db.query("DELETE FROM client_tumblr WHERE client_id = $1", [
        client_id,
      ]);
      await redis.del(`client_tumblr:${client_id}`);
      await redis.del("client_tumblr:all");
      return { message: "Client record deleted successfully" };
    } catch (err) {
      console.error("❌ Error deleting client record:", err);
      throw err;
    }
  }
}

module.exports = ClientTumblr;
