const db = require("../../config/db");
const redis = require("../../config/redis");

class ClientLinkedIn {
  // 🟢 Create client_linkedin Table
  static async createTable() {
    const query = `
      CREATE TABLE IF NOT EXISTS client_linkedin (
        client_id INT PRIMARY KEY,
        sales_channel_id INT NOT NULL REFERENCES master_sales_channel(channel_id) ON DELETE CASCADE,
        organisation_id TEXT NOT NULL,
        access_token TEXT NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      );
    `;

    try {
      await db.query(query);
      console.log("✅ client_linkedin table is ready.");
    } catch (err) {
      console.error("❌ Error creating client_linkedin table:", err);
    }
  }

  // 🟢 Create a new client record
  static async create(clientData) {
    const query = `
      INSERT INTO client_linkedin (
        client_id, sales_channel_id, organisation_id, access_token
      ) VALUES ($1, $2, $3, $4)
      RETURNING client_id;
    `;

    const values = [
      clientData.client_id,
      clientData.sales_channel_id,
      clientData.organisation_id,
      clientData.access_token,
    ];

    try {
      const { rows } = await db.query(query, values);
      await redis.del("client_linkedin:all"); // Invalidate cached list
      return rows[0].client_id;
    } catch (err) {
      console.error("❌ Error inserting client record:", err);
      throw err;
    }
  }

  // 🟢 Get all clients (with caching)
  static async getAll() {
    const cachedClients = await redis.get("client_linkedin:all");
    if (cachedClients) {
      return JSON.parse(cachedClients);
    }

    try {
      const { rows } = await db.query("SELECT * FROM client_linkedin");
      await redis.setex("client_linkedin:all", 3600, JSON.stringify(rows)); // Cache for 1 hour
      return rows;
    } catch (err) {
      console.error("❌ Error fetching clients:", err);
      throw err;
    }
  }

  // 🟢 Get client by ID (with caching)
  static async getById(client_id) {
    const cachedClient = await redis.get(`client_linkedin:${client_id}`);
    if (cachedClient) {
      return JSON.parse(cachedClient);
    }

    try {
      const {
        rows,
      } = await db.query("SELECT * FROM client_linkedin WHERE client_id = $1", [
        client_id,
      ]);

      if (rows.length === 0) return null;

      await redis.setex(
        `client_linkedin:${client_id}`,
        3600,
        JSON.stringify(rows[0])
      ); // Cache for 1 hour
      return rows[0];
    } catch (err) {
      console.error("❌ Error fetching client by ID:", err);
      throw err;
    }
  }

  // 🟢 Update a client record
  static async update(clientData) {
    const query = `
      UPDATE client_linkedin
      SET sales_channel_id = $2, organisation_id = $3, access_token = $4
      WHERE client_id = $1
      RETURNING client_id;
    `;

    const values = [
      clientData.client_id,
      clientData.sales_channel_id,
      clientData.organisation_id,
      clientData.access_token,
    ];

    try {
      const { rows } = await db.query(query, values);
      await redis.del(`client_linkedin:${clientData.client_id}`);
      await redis.del("client_linkedin:all"); // Invalidate cached list
      return rows[0].client_id;
    } catch (err) {
      console.error("❌ Error updating client record:", err);
      throw err;
    }
  }

  // 🟢 Delete a client record
  static async delete(client_id) {
    try {
      await db.query("DELETE FROM client_linkedin WHERE client_id = $1", [
        client_id,
      ]);
      await redis.del(`client_linkedin:${client_id}`);
      await redis.del("client_linkedin:all"); // Invalidate cached list
      return { message: "Client record deleted successfully" };
    } catch (err) {
      console.error("❌ Error deleting client record:", err);
      throw err;
    }
  }
}

module.exports = ClientLinkedIn;
