const db = require("../../config/db");
const redis = require("../../config/redis");

class ClientAmazonIndia {
  // 🟢 Create client_amazon_india Table
  static async createTable() {
    const query = `
      CREATE TABLE IF NOT EXISTS client_amazon_india (
        client_id INT PRIMARY KEY,
        sales_channel_id INT NOT NULL REFERENCES master_sales_channel(channel_id) ON DELETE CASCADE,
        amz_client_id TEXT NOT NULL,
        amz_client_secret TEXT NOT NULL,
        amz_app_id TEXT NOT NULL,
        amz_refresh_token TEXT NOT NULL,
        lwa_app_id TEXT NOT NULL,
        lwa_client_secret TEXT NOT NULL,
        aws_secret_key TEXT NOT NULL,
        aws_access_key TEXT NOT NULL,
        marketplace_id TEXT NOT NULL,
        seller_id TEXT NOT NULL,
        auth_url TEXT NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      );
    `;

    try {
      await db.query(query);
      console.log("✅ client_amazon_india table is ready.");
    } catch (err) {
      console.error("❌ Error creating client_amazon_india table:", err);
    }
  }

  // 🟢 Create a new client record (client_id is passed manually)
  static async create(clientData) {
    const query = `
      INSERT INTO client_amazon_india (
        client_id, sales_channel_id, amz_client_id, amz_client_secret, amz_app_id, 
        amz_refresh_token, lwa_app_id, lwa_client_secret, aws_secret_key, 
        aws_access_key, marketplace_id, seller_id, auth_url
      ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13)
      RETURNING client_id;
    `;

    const values = [
      clientData.client_id,
      clientData.sales_channel_id,
      clientData.amz_client_id,
      clientData.amz_client_secret,
      clientData.amz_app_id,
      clientData.amz_refresh_token,
      clientData.lwa_app_id,
      clientData.lwa_client_secret,
      clientData.aws_secret_key,
      clientData.aws_access_key,
      clientData.marketplace_id,
      clientData.seller_id,
      clientData.auth_url,
    ];

    try {
      const { rows } = await db.query(query, values);
      await redis.del("clients_india:all"); // Invalidate cached list
      return rows[0].client_id;
    } catch (err) {
      console.error("❌ Error inserting client record:", err);
      throw err;
    }
  }

  // 🟢 Get all client records (with caching)
  static async getAll() {
    const cachedClients = await redis.get("clients_india:all");
    if (cachedClients) {
      return JSON.parse(cachedClients);
    }

    try {
      const { rows } = await db.query("SELECT * FROM client_amazon_india");
      await redis.setex("clients_india:all", 3600, JSON.stringify(rows)); // Cache for 1 hour
      return rows;
    } catch (err) {
      console.error("❌ Error fetching all clients:", err);
      throw err;
    }
  }

  // 🟢 Get a client record by ID (with caching)
  static async getById(client_id) {
    const cachedClient = await redis.get(`clients_india:${client_id}`);
    if (cachedClient) {
      return JSON.parse(cachedClient);
    }

    try {
      const {
        rows,
      } = await db.query(
        "SELECT * FROM client_amazon_india WHERE client_id = $1",
        [client_id]
      );

      if (rows.length === 0) return null;

      await redis.setex(
        `clients_india:${client_id}`,
        3600,
        JSON.stringify(rows[0])
      ); // Cache for 1 hour
      return rows[0];
    } catch (err) {
      console.error("❌ Error fetching client by ID:", err);
      throw err;
    }
  }

  // 🟢 Update a client record
  static async update(clientData) {
    const query = `
      UPDATE client_amazon_india
      SET sales_channel_id = $2, amz_client_id = $3, amz_client_secret = $4, 
          amz_app_id = $5, amz_refresh_token = $6, lwa_app_id = $7, lwa_client_secret = $8, 
          aws_secret_key = $9, aws_access_key = $10, marketplace_id = $11, 
          seller_id = $12, auth_url = $13
      WHERE client_id = $1
      RETURNING client_id;
    `;

    const values = [
      clientData.client_id,
      clientData.sales_channel_id,
      clientData.amz_client_id,
      clientData.amz_client_secret,
      clientData.amz_app_id,
      clientData.amz_refresh_token,
      clientData.lwa_app_id,
      clientData.lwa_client_secret,
      clientData.aws_secret_key,
      clientData.aws_access_key,
      clientData.marketplace_id,
      clientData.seller_id,
      clientData.auth_url,
    ];

    try {
      const { rows } = await db.query(query, values);
      await redis.del(`clients_india:${clientData.client_id}`);
      await redis.del("clients_india:all"); // Invalidate cached list
      return rows[0].client_id;
    } catch (err) {
      console.error("❌ Error updating client record:", err);
      throw err;
    }
  }

  // 🟢 Delete a client record
  static async delete(client_id) {
    try {
      await db.query("DELETE FROM client_amazon_india WHERE client_id = $1", [
        client_id,
      ]);

      await redis.del(`clients:${client_id}`);
      await redis.del("clients:all"); // Invalidate cached list
      return { message: "Client record deleted successfully" };
    } catch (err) {
      console.error("❌ Error deleting client record:", err);
      throw err;
    }
  }
}

module.exports = ClientAmazonIndia;
