const db = require("../../config/db");
const redis = require("../../config/redis");

class MasterPayment {
  static async createTable() {
    const query = `
      CREATE TABLE IF NOT EXISTS master_payment (
        id SERIAL PRIMARY KEY,
        payment_gateway_name VARCHAR(255) NOT NULL UNIQUE,
        url TEXT NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      );
    `;

    try {
      await db.query(query);
      console.log("✅ master_payment table is ready.");
    } catch (err) {
      console.error("❌ Error creating master_payment table:", err);
    }
  }

  static async create({ payment_gateway_name, url }) {
    const query = `
      INSERT INTO master_payment (payment_gateway_name, url)
      VALUES ($1, $2)
      RETURNING *;
    `;

    try {
      const { rows } = await db.query(query, [payment_gateway_name, url]);
      await redis.del("payments:all"); // Invalidate cached list
      return rows[0];
    } catch (err) {
      console.error("❌ Error inserting payment gateway:", err);
      throw err;
    }
  }

  static async getAll() {
    const cachedPayments = await redis.get("payments:all");
    if (cachedPayments) {
      return JSON.parse(cachedPayments);
    }

    const { rows } = await db.query("SELECT * FROM master_payment");
    await redis.setex("payments:all", 3600, JSON.stringify(rows)); // Cache for 1 hour
    return rows;
  }

  static async getById(id) {
    const cachedPayment = await redis.get(`payments:${id}`);
    if (cachedPayment) {
      return JSON.parse(cachedPayment);
    }

    const {
      rows,
    } = await db.query("SELECT * FROM master_payment WHERE id = $1", [id]);
    if (rows.length === 0) return null;

    await redis.setex(`payments:${id}`, 3600, JSON.stringify(rows[0]));
    return rows[0];
  }

  static async update({ id, payment_gateway_name, url }) {
    const query = `
      UPDATE master_payment 
      SET payment_gateway_name=$2, url=$3
      WHERE id=$1
      RETURNING *;
    `;

    try {
      const { rows } = await db.query(query, [id, payment_gateway_name, url]);
      await redis.del(`payments:${id}`);
      await redis.del("payments:all");
      return rows[0];
    } catch (err) {
      console.error("❌ Error updating payment gateway:", err);
      throw err;
    }
  }

  static async delete(id) {
    await db.query("DELETE FROM master_payment WHERE id = $1", [id]);
    await redis.del(`payments:${id}`);
    await redis.del("payments:all");
    return { message: "Payment gateway deleted successfully" };
  }
}

module.exports = MasterPayment;
