const db = require("../config/db");
const redis = require("../config/redis");

class ClientUsers {
  static async createTable() {
    const query = `
      CREATE TABLE IF NOT EXISTS client_users (
        client_id INT NOT NULL,
        loc_id INT NOT NULL,
        user_entity_id SERIAL NOT NULL,
        first_name TEXT NOT NULL,
        last_name TEXT NOT NULL,
        user_name VARCHAR(255) UNIQUE NOT NULL,
        password VARCHAR(255) NOT NULL,
        user_type VARCHAR(50),
        work_email VARCHAR(255) UNIQUE,
        personal_email VARCHAR(255),
        telephone VARCHAR(50),
        user_status INT,
        PRIMARY KEY (client_id, loc_id, user_entity_id),
        FOREIGN KEY (client_id, loc_id) REFERENCES client_locations(client_id, loc_id) ON DELETE CASCADE
      );
    `;

    try {
      await db.query(query);
      console.log("✅ client_users table is ready.");
    } catch (err) {
      console.error("❌ Error creating client_users table:", err);
    }
  }

  static async create({
    client_id,
    loc_id,
    first_name,
    last_name,
    user_name,
    password,
    user_type,
    work_email,
    personal_email,
    telephone,
    user_status,
  }) {
    const query = `
      INSERT INTO client_users (client_id, loc_id, first_name, last_name, user_name, password, user_type, work_email, personal_email, telephone, user_status)
      VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11)
      RETURNING user_entity_id, client_id, loc_id;
    `;

    const values = [
      client_id,
      loc_id,
      first_name,
      last_name,
      user_name,
      password,
      user_type,
      work_email,
      personal_email,
      telephone,
      user_status,
    ];

    try {
      const { rows } = await db.query(query, values);
      await redis.del("client_users:all");
      await redis.del(`client_users:${client_id}`);
      return rows[0]; // Returns created user details
    } catch (err) {
      console.error("❌ Error inserting client user:", err);
      throw err;
    }
  }

  static async getAll() {
    const cachedUsers = await redis.get("client_users:all");
    if (cachedUsers) {
      return JSON.parse(cachedUsers);
    }

    const query = "SELECT * FROM client_users";
    try {
      const { rows } = await db.query(query);
      await redis.setex("client_users:all", 3600, JSON.stringify(rows));
      return rows;
    } catch (err) {
      console.error("❌ Error fetching client users:", err);
      throw err;
    }
  }

  static async getByClientId(client_id) {
    const cachedUsers = await redis.get(`client_users:${client_id}`);
    if (cachedUsers) {
      return JSON.parse(cachedUsers);
    }

    const query = "SELECT * FROM client_users WHERE client_id = $1";
    try {
      const { rows } = await db.query(query, [client_id]);
      await redis.setex(
        `client_users:${client_id}`,
        3600,
        JSON.stringify(rows)
      );
      return rows;
    } catch (err) {
      console.error("❌ Error fetching client users by client_id:", err);
      throw err;
    }
  }

  static async update({
    user_entity_id,
    client_id,
    loc_id,
    first_name,
    last_name,
    user_name,
    password,
    user_type,
    work_email,
    personal_email,
    telephone,
    user_status,
  }) {
    const query = `
      UPDATE client_users
      SET first_name = $4, last_name = $5, user_name = $6, password = $7, user_type = $8,
          work_email = $9, personal_email = $10, telephone = $11, user_status = $12
      WHERE user_entity_id = $1 AND client_id = $2 AND loc_id = $3
      RETURNING user_entity_id, client_id, loc_id;
    `;

    const values = [
      user_entity_id,
      client_id,
      loc_id,
      first_name,
      last_name,
      user_name,
      password,
      user_type,
      work_email,
      personal_email,
      telephone,
      user_status,
    ];

    try {
      const { rows } = await db.query(query, values);
      await redis.del("client_users:all");
      await redis.del(`client_users:${client_id}`);
      return rows[0];
    } catch (err) {
      console.error("❌ Error updating client user:", err);
      throw err;
    }
  }

  static async delete(client_id, loc_id, user_entity_id) {
    const query = `
      DELETE FROM client_users 
      WHERE client_id = $1 AND loc_id = $2 AND user_entity_id = $3
      RETURNING user_entity_id;
    `;

    try {
      const { rows } = await db.query(query, [
        client_id,
        loc_id,
        user_entity_id,
      ]);

      if (rows.length === 0) {
        return { message: "User not found" };
      }

      await redis.del("client_users:all");
      await redis.del(`client_users:${client_id}`);

      return { message: "Client user deleted successfully" };
    } catch (err) {
      console.error("❌ Error deleting client user:", err);
      throw err;
    }
  }
}

module.exports = ClientUsers;
