const db = require("../config/db");
const redis = require("../config/redis");

class ClientLocationModel {
  static async createTable() {
    const query = `
      CREATE TABLE IF NOT EXISTS client_locations (
        client_id INT NOT NULL,
        loc_id SERIAL NOT NULL,
        client_type_id INT NOT NULL,
        fax VARCHAR(255),
        locationCity VARCHAR(255),
        locationDistName VARCHAR(255),
        locationName VARCHAR(255),
        phone VARCHAR(50),
        postCode VARCHAR(20),
        streetLine1 VARCHAR(255),
        streetLine2 VARCHAR(255),
        locationStatus INT,
        PRIMARY KEY (client_id, loc_id),
        FOREIGN KEY (client_id) REFERENCES clients(client_id) ON DELETE CASCADE
      );
    `;

    try {
      await db.query(query);
      console.log("✅ client_locations table is ready.");
    } catch (err) {
      console.error("❌ Error creating client_locations table:", err);
    }
  }

  static async create({
    client_id,
    client_type_id,
    fax,
    locationCity,
    locationDistName,
    locationName,
    phone,
    postCode,
    streetLine1,
    streetLine2,
    locationStatus,
  }) {
    const query = `
      INSERT INTO client_locations (client_id, client_type_id, fax, locationCity, locationDistName, locationName, phone, postCode, streetLine1, streetLine2, locationStatus)
      VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11)
      RETURNING client_id, loc_id;
    `;

    const values = [
      client_id,
      client_type_id,
      fax,
      locationCity,
      locationDistName,
      locationName,
      phone,
      postCode,
      streetLine1,
      streetLine2,
      locationStatus,
    ];

    try {
      const { rows } = await db.query(query, values);
      await redis.del(`client_locations:all`);
      await redis.del(`client_locations:${client_id}`);
      return rows[0]; // Returns client_id and the auto-generated loc_id
    } catch (err) {
      console.error("❌ Error inserting client location:", err);
      throw err;
    }
  }

  static async getAll() {
    const cachedLocations = await redis.get("client_locations:all");
    if (cachedLocations) {
      return JSON.parse(cachedLocations);
    }

    const query = "SELECT * FROM client_locations";
    const { rows } = await db.query(query);
    await redis.setex("client_locations:all", 3600, JSON.stringify(rows));
    return rows;
  }

  static async getByClientId(client_id) {
    const cachedLocations = await redis.get(`client_locations:${client_id}`);
    if (cachedLocations) {
      return JSON.parse(cachedLocations);
    }

    const query = "SELECT * FROM client_locations WHERE client_id = $1";
    const { rows } = await db.query(query, [client_id]);
    await redis.setex(
      `client_locations:${client_id}`,
      3600,
      JSON.stringify(rows)
    );
    return rows;
  }

  static async update({
    client_id,
    loc_id,
    client_type_id,
    fax,
    locationCity,
    locationDistName,
    locationName,
    phone,
    postCode,
    streetLine1,
    streetLine2,
    locationStatus,
  }) {
    const query = `
      UPDATE client_locations
      SET client_type_id =$3, fax = $4, locationCity = $5, locationDistName = $6, locationName = $7,
          phone = $8, postCode = $9, streetLine1 = $10, streetLine2 = $11, locationStatus = $12
      WHERE client_id = $1 AND loc_id = $2
      RETURNING client_id, loc_id;
    `;

    const values = [
      client_id,
      loc_id,
      client_type_id,
      fax,
      locationCity,
      locationDistName,
      locationName,
      phone,
      postCode,
      streetLine1,
      streetLine2,
      locationStatus,
    ];

    try {
      const { rows } = await db.query(query, values);
      await redis.del(`client_locations:all`);
      await redis.del(`client_locations:${client_id}`);
      return rows[0];
    } catch (err) {
      console.error("❌ Error updating client location:", err);
      throw err;
    }
  }

  static async delete(client_id, loc_id) {
    const query =
      "DELETE FROM client_locations WHERE client_id = $1 AND loc_id = $2";
    try {
      await db.query(query, [client_id, loc_id]);
      await redis.del(`client_locations:all`);
      await redis.del(`client_locations:${client_id}`);
      return { message: "Client location deleted successfully" };
    } catch (err) {
      console.error("❌ Error deleting client location:", err);
      throw err;
    }
  }
}

module.exports = ClientLocationModel;
